<?php  if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Query categories data in the database
 *
 * @copyright   Copyright (C) 2018, Echo Plugins
 */
class EPKB_Categories_DB {

	/**
	 * Get all top-level categories
	 *
	 * @param $kb_id
	 * @param string $hide_choice - if 'hide_empty' then do not return empty categories
	 *
	 * @return array or empty array on error
	 *
	 */
	static function get_top_level_categories( $kb_id, $hide_choice='hide_empty' ) {

		if ( ! EPKB_Utilities::is_positive_int( $kb_id ) ) {
			EPKB_Logging::add_log( 'Invalid kb id', $kb_id );
			return array();
		}

		$args = array(
				'parent'        => '0',
				'hide_empty'    => $hide_choice === 'hide_empty' // whether to return categories without articles
		);

		$terms = get_terms( EPKB_KB_Handler::get_category_taxonomy_name( $kb_id ), $args );
		if ( is_wp_error( $terms ) ) {
			EPKB_Logging::add_log( 'cannot get terms for kb_id', $kb_id, $terms );
			return array();
		} else if ( empty($terms) || ! is_array($terms) ) {
			return array();
		}

		return array_values($terms);   // rearrange array keys
	}

	/**
	 * Show list of top or sibling KB Categories, each with link to Category Archive page and total article count
	 *
	 * @param $kb_id
	 * @param $kb_config
	 * @param $parent_id
	 * @param $active_id
	 * @return string
	 */
	static function get_layout_categories_list( $kb_id, $kb_config, $parent_id = 0, $active_id = 0 ) {

		$is_demo_data = ! empty($_POST['epkb-wizard-demo-data']);

		// determine what categories will be displayed in the Category Focused Layout list
		if ( $is_demo_data ) {
			$top_categories = EPKB_KB_Demo_Data::get_demo_categories_list();
		} else if ( ! $parent_id || ( isset( $kb_config['categories_layout_list_mode'] ) && ( $kb_config['categories_layout_list_mode'] == 'list_top_categories' ) ) ) {
			$top_categories = self::get_top_level_categories( $kb_id );
		} else {
			$top_categories = self::get_child_categories( $kb_id, $parent_id );
		}
		
		if ( empty($top_categories) ) {
			return '';
		}

		$title                      = 'Categories';  //TODO Get KB setting
		$title_text_color           = '#666666';     //TODO Get KB setting
		$container_background_color = '#fcfcfc';     //TODO Get KB setting
		$container_margin_top       = '0';         //TODO Get KB setting
		$font_size                  = '14';          //TODO Get KB setting
		$category_text_color        = '#2b98e5 ';    //TODO Get KB setting

		//Count Styling
		$count_background_color     = '#FFFFFF ';    //TODO Get KB setting
		$count_text_color           = '#000000';     //TODO Get KB setting
		$count_border_color         = '#CCCCCC';     //TODO Get KB setting


		$article_db = new EPKB_Articles_DB();
		ob_start();                         		?>

		<style>
			.eckb-acll__title {
				color:<?php echo $title_text_color; ?>;
			}
			.eckb-article-cat-layout-list {
				background-color:<?php echo $container_background_color; ?>;
				font-size:<?php echo $font_size; ?>px;
				margin-top:<?php echo $container_margin_top; ?>px;
			}
			.eckb-article-cat-layout-list a {
				font-size:<?php echo $font_size; ?>px;
			}
			.eckb-acll__cat-item__name {
				color:<?php echo $category_text_color; ?>;
			}
			.eckb-acll__cat-item__count {
				color:<?php echo $count_text_color; ?>;
				background-color:<?php echo $count_background_color; ?>;
				border:solid 1px <?php echo $count_border_color; ?>;
			}
		</style>

		<div class="eckb-article-cat-layout-list eckb-article-cat-layout-list-reset">
			<div class="eckb-article-cat-layout-list__inner">
				<div class="eckb-acll__title"><?php echo $title; ?></div>
				<ul>						<?php

					// display each category in a list
					foreach( $top_categories as $top_category ) {
						
						if ( $is_demo_data ) {
							$top_category = (object) $top_category;
							$term_link = '#';
							$active = $top_category->active;
							$count = $top_category->count;
						} else {
							$term_link = get_term_link( $top_category, $top_category->taxonomy );
							if ( empty($term_link) || is_wp_error( $term_link ) ) {
								$term_link = '';
							}

							$active = ! empty($active_id) && $active_id == $top_category->term_id;
							
							$count = count($article_db->get_published_articles_by_sub_or_category( $kb_id, $top_category->term_id, 'date', -1, true ) );
						}	?>

						<li class="eckb--acll__cat-item <?php echo $active ? 'eckb--acll__cat-item--active' : ''; ?>">
							<a href="<?php echo $term_link; ?>">
								<div>
									<span class="eckb-acll__cat-item__name">
										<?php echo $top_category->name; ?>
									</span>
								</div>
								<div>
									<span class="eckb-acll__cat-item__count">
										<?php echo $count; ?>
									</span>
								</div>
							</a>
						</li>						<?php
					}	?>

				</ul>
			</div>
		</div>			<?php

		return ob_get_clean();
	}

	/**
	 * Get all categories that belong to given parent
	 *
	 * @param $kb_id
	 * @param int $parent_id is parent category we use to find children
	 * @param string $hide_choice
	 *
	 * @return array or empty array on error
	 */
	static function get_child_categories( $kb_id, $parent_id, $hide_choice = 'hide_empty' ) {

		if ( ! EPKB_Utilities::is_positive_int( $kb_id ) ) {
			EPKB_Logging::add_log( 'Invalid kb id', $kb_id );
			return array();
		}

		if ( ! EPKB_Utilities::is_positive_int( $parent_id ) ) {
			EPKB_Logging::add_log( 'Invalid parent id', $parent_id );
			return array();
		}

		$args = array(
				'child_of'      => $parent_id,
				'parent'        => $parent_id,
				'hide_empty'    => $hide_choice === 'hide_empty'
		);

		$terms = get_terms( EPKB_KB_Handler::get_category_taxonomy_name( $kb_id ), $args );
		if ( is_wp_error( $terms ) ) {
			EPKB_Logging::add_log( 'failed to get terms for kb_id: ' . $kb_id . ', parent_id: ' . $parent_id, $terms );
			return array();
		}

		if ( empty( $terms ) || ! is_array( $terms ) ) {
			return array();
		}

		return array_values($terms);
	}
}