<?php  if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Handle article front-end setup
 *
 */
class EPKB_Articles_Setup {

	private $cached_comments_flag;

	public function __construct() {
		add_filter( 'comments_open', array( $this, 'setup_comments'), 1, 2 );
	}

    /**
     * Output SBL + article
     *
     * @param $article_content - article + features
     * @param $kb_config
     * @param bool $is_builder_on
     * @param array $article_seq
     * @param array $categories_seq
     * @return string
     */
    public static function output_article_page_with_layout( $article_content, $kb_config, $is_builder_on=false, $article_seq=array(), $categories_seq=array() ) {

        // get Article Page Layout
        ob_start();
        apply_filters( 'epkb_article_page_layout_output', $article_content, $kb_config, $is_builder_on, $article_seq, $categories_seq );
        $layout_output = ob_get_clean();

        // if no layout found then just display the article
        if ( empty($layout_output) ) {
            $layout_output = $article_content;
        }

        return $layout_output;
	}

    /**
     * Return single article content surrounded by features like breadcrumb and tags.
     *
     * NOTE: Assumes shortcodes already ran.
     *
     * @param $article
     * @param $content
     * @param $kb_config - front end or back end temporary KB config
     * @return string
     */
	public static function get_article_content_and_features( $article, $content, $kb_config ) {

		global $epkb_password_checked;

		if ( empty($epkb_password_checked) && post_password_required() ) {
			return get_the_password_form();
		}

        // if global post is empty initialize it
        if ( empty($GLOBALS['post']) ) {
            $GLOBALS['post'] = $article;
        }

        // if necessary get KB configuration
        if ( empty($kb_config) ) {
            $kb_id = EPKB_KB_Handler::get_kb_id_from_post_type( $article->post_type );
            if ( is_wp_error($kb_id) ) {
                $kb_id = EPKB_KB_Config_DB::DEFAULT_KB_ID;
            }

            // initialize KB config to be accessible to templates
            $kb_config = epkb_get_instance()->kb_config_obj->get_kb_config_or_default( $kb_id );
        }

		// setup article structure - either old version 1 or new version 2

		self::setup_article_content_hooks( $kb_config );

        $article_page_container_classes = apply_filters( 'eckb-article-page-container-classes', array(), $kb_config['id'] );  // used for old Widgets KB Sidebar
        $article_page_container_classes = isset($article_page_container_classes) && is_array($article_page_container_classes) ? $article_page_container_classes : array();

		ob_start();

		// Article Structure
		$article_container_structure_version = 'eckb-article-page-container';
		if ( self::is_article_structure_v2( $kb_config ) ) {
			$article_container_structure_version = 'eckb-article-page-container-v2';
			self::generate_article_structure_css_v2( $kb_config );
		}				?>

        <div id="<?php echo $article_container_structure_version; ?>" class="<?php echo implode(" ", $article_page_container_classes); ?>" >    <?php

            self::article_section( 'eckb-article-header', array( 'id' => $kb_config['id'], 'config' => $kb_config, 'article' => $article ) ); ?>

            <div id="eckb-article-body">  <?php

                self::article_section( 'eckb-article-left-sidebar', array( 'id' => $kb_config['id'], 'config' => $kb_config, 'article' => $article ) ); ?>

                <div id="eckb-article-content">                        <?php

                    self::article_section( 'eckb-article-content-header', array( 'id' => $kb_config['id'], 'config' => $kb_config, 'article' => $article ) );
                    self::article_section( 'eckb-article-content-body',   array( 'id' => $kb_config['id'], 'config' => $kb_config, 'article' => $article, 'content' => $content ) );
                    self::article_section( 'eckb-article-content-footer', array( 'id' => $kb_config['id'], 'config' => $kb_config, 'article' => $article ) );                        ?>

                </div><!-- /#eckb-article-content -->     <?php

                self::article_section( 'eckb-article-right-sidebar', array( 'id' => $kb_config['id'], 'config' => $kb_config, 'article' => $article ) ); ?>

            </div><!-- /#eckb-article-body -->              <?php

            self::article_section( 'eckb-article-footer', array( 'id' => $kb_config['id'], 'config' => $kb_config, 'article' => $article ) ); ?>

        </div><!-- /#eckb-article-page-container -->        <?php

		$article_content = ob_get_clean();

        return str_replace( ']]>', ']]&gt;', $article_content );
	}

    /**
     * Call all hooks for given article section.
     *
     * @param $hook - both hook name and div id
     * @param $args
     */
	public static function article_section( $hook, $args ) {
        echo '<div id="' . $hook . '">';
        do_action( $hook, $args );
        echo '</div>';
	}

	/**
	 * REGISTER all article hooks we need
	 *
	 * @param $kb_config
	 */
    private static function setup_article_content_hooks( $kb_config ) {


    	// CONTENT HEADER
        add_action( 'eckb-article-content-header', array('EPKB_Articles_Setup', 'article_title'), 9, 3 );
        add_action( 'eckb-article-content-header', array('EPKB_Articles_Setup', 'meta_container_header'), 9, 3 );
        add_action( 'eckb-article-content-header', array('EPKB_Articles_Setup', 'breadcrumbs'), 9, 3 );
        add_action( 'eckb-article-content-header', array('EPKB_Articles_Setup', 'navigation'), 9, 3 );

        // LEFT SIDEBAR
	    add_action( 'eckb-article-left-sidebar', array('EPKB_Articles_Setup', 'focused_layout_categories'), 10, 3 );

	    // BODY
        add_action( 'eckb-article-content-body', array('EPKB_Articles_Setup', 'article_content'), 10, 4 );

        //TOC Needs to be styled with CSS differently based on Article version. --------------

	    // Version 2: We can place the TOC into their approperient HTML containers ( Left Sidebar , Main Content , Right Sidebar )
	    if ( self::is_article_structure_v2( $kb_config ) ) {
		    if ( $kb_config['article_toc_position'] === 'right' ) {
			    add_action( 'eckb-article-right-sidebar', array('EPKB_Articles_Setup', 'table_of_content'), 10, 3 );
		    } elseif ( $kb_config['article_toc_position'] === 'left' ) {
			    add_action( 'eckb-article-left-sidebar', array('EPKB_Articles_Setup', 'table_of_content'), 10, 3 );
		    } elseif ( $kb_config['article_toc_position'] === 'middle' ) {
			    add_action( 'eckb-article-content-header', array('EPKB_Articles_Setup', 'table_of_content'), 10, 3 );
		    }
	    }
	    // Version 1: We need to insert the TOC into the Article Body.
	    else {
		    add_action( 'eckb-article-content-body', array('EPKB_Articles_Setup', 'table_of_content'), 8, 4 );
	    }

	    // CONTENT FOOTER
	    add_action( 'eckb-article-content-footer', array('EPKB_Articles_Setup', 'meta_container_footer'), 10, 3 );

	    add_action( 'eckb-article-footer', array('EPKB_Articles_Setup', 'tags'), 10, 3 );
        add_action( 'eckb-article-footer', array('EPKB_Articles_Setup', 'comments'), 10, 3 );
    }

    // ARTICLE TITLE
    public static function article_title( $args ) {
        $show_title = $args['config']['templates_for_kb'] == 'kb_templates';
        $article_title = $show_title ? get_the_title( $args['article'] ) : '';

		if ( isset($_POST['epkb-wizard-demo-data']) && $_POST['epkb-wizard-demo-data'] == true ) {
			$article_title =  __( 'Demo Article', 'echo-knowledge-base' );
		}

        $tag = $show_title ? 'h1' : 'div';
        $article_seq_no = empty($_REQUEST['seq_no']) ? '' : EPKB_Utilities::sanitize_int( $_REQUEST['seq_no'] );
        $article_seq_no = empty($article_seq_no) ? '' : ' data-kb_article_seq_no=' . $article_seq_no;
        echo '<' . $tag . ' class="eckb-article-title kb-article-id" id="' . $args['article']->ID . '"' . $article_seq_no . '>' . $article_title . '</' . $tag . '>';
    }

	public static function meta_container_header( $args ) {
		self::output_meta_container( $args, 'header' );
	}

	public static function meta_container_footer( $args ) {
		self::output_meta_container( $args, 'footer' );
	}

	private static function output_meta_container( $args, $location ) {

		$args['output_location'] = $location == 'header' ? 'top' : 'bottom';
		$args['is_meta_container_on'] = false;

		$check_value = 'article_' . ( $location == 'header' ? 'top' : 'bottom' );
		$is_created_on      = $args['config']['created_on'] == $check_value;
		$is_last_updated_on = $args['config']['last_udpated_on'] == $check_value;
		$is_author          = $args['config']['author_mode'] == $check_value;

		$data = apply_filters( 'eckb-article-meta-container-end-on', $args );
		$is_meta_on = empty($data['is_meta_container_on']) ? false : $data['is_meta_container_on'];

		if ( ! $is_created_on && ! $is_last_updated_on && ! $is_author && ! $is_meta_on ) {
			return;
		}

		/** below is class eckb-article-content-header__article-meta and eckb-article-content-footer__article-meta */
		echo '<div class="' . 'eckb-article-content-' . $location . '__article-meta' . '">';

		if ( $is_created_on ) {
			self::created_on( $args );
		}

		if ( $is_last_updated_on ) {
			self::last_updated_on( $args );
		}

		if ( $is_author ) {
			self::author( $args );
		}

		// add Hook for future meta to inject here.
		do_action( 'eckb-article-meta-container-end', $args );

		echo '</div>';
	}

	/**
	 * Output Table of Content
	 *
	 * @param $args
	 */
	public static function table_of_content( $args ) {

		// check toc settings and show toc container if need
		if ( 'on' != $args['config']['article_toc_enable'] ) {
			return;
		}

		$media_1_gutter = $args['config']['article_toc_width_1']+$args['config']['article_toc_gutter'] + 25;
		$media_2_gutter = $args['config']['article_toc_width_2']+$args['config']['article_toc_gutter'] + 25;
		$toc_gutter = -325 - $args['config']['article_toc_gutter'];

		// Both Versions
		$styles = '
				#eckb-article-body .eckb-article-toc {
					top: ' . $args['config']['article_toc_position_from_top'] . 'px;
				}
				#eckb-article-body .eckb-article-toc ul a.active {
					background-color:   ' . $args['config']['article_toc_active_bg_color'] . ';
					color:              ' . $args['config']['article_toc_active_text_color'] . ';
				}
				#eckb-article-body .eckb-article-toc ul a:hover {
					background-color:   ' . $args['config']['article_toc_cursor_hover_bg_color'] . ';
					color:              ' . $args['config']['article_toc_cursor_hover_text_color'] . ';
				}
				#eckb-article-body .eckb-article-toc__inner {
					border-color: ' . $args['config']['article_toc_border_color'] . ';
					font-size:          ' . $args['config']['article_toc_font_size'] . 'px;
					background-color:   ' . $args['config']['article_toc_background_color'] . ';
				}
				#eckb-article-body .eckb-article-toc__inner a {
					color:              ' . $args['config']['article_toc_text_color'] . ';
					font-size:          ' . $args['config']['article_toc_font_size'] . 'px;
				}
		';

		// If Article Version 2 we don't need some of these TOC Settings anymore.
		if ( ! self::is_article_structure_v2( $args['config'] ) ) {

			// Version 1 Only
			$styles .= '
			#eckb-article-body .eckb-article-toc--position-'.$args['config']['article_toc_position'].' .eckb-article-toc__inner {
					'.$args['config']['article_toc_position'].': ' . $toc_gutter . 'px;
				}
			';

			// Media Queries
			$styles .= '
				@media only screen and ( max-width: ' . $args['config']['article_toc_media_1'] . 'px ) {
                    #eckb-article-body .eckb-article-toc .eckb-article-toc__inner {
                        width:          ' . $args['config']['article_toc_width_1'] . 'px !important;
                        '.$args['config']['article_toc_position'].': -' . $media_1_gutter . 'px !important;
                    }
                }
                @media only screen and ( max-width: ' . $args['config']['article_toc_media_2'] . 'px ) {
                    #eckb-article-body .eckb-article-toc .eckb-article-toc__inner {
                        width:          ' . $args['config']['article_toc_width_2'] . 'px !important;
                        '.$args['config']['article_toc_position'].': -' . $media_2_gutter . 'px !important;
                    }
                }
                @media only screen and ( max-width: ' . $args['config']['article_toc_media_3'] . 'px ) {
                
                    #eckb-article-body .eckb-article-toc {
                          width:100% !important;
					      float: none !important;
					      position: relative !important;
					      height: auto !important;
					      height:fit-content !important;
                          display:inline-block !important;
                          top:0 !important;
                    }
                    #eckb-article-body .eckb-article-toc .eckb-article-toc__inner {
                          display:block !important;
					      width:100% !important;
					      float: none !important;
					      margin-bottom: 20px !important;
					      left: 0 !important;
					      position: relative !important;
					      
                    }
                } ';
		}

		$classes = 'eckb-article-toc--position-' . $args['config']['article_toc_position'];
		$classes .= ' eckb-article-toc--bmode-' . $args['config']['article_toc_border_mode'];

		$title = empty($args['config']['article_toc_title']) ? '' : '<div class="eckb-article-toc__title">' . $args['config']['article_toc_title'] . '</div>';

		$wrap = '
			<style type="text/css">' . $styles . '</style>
			<div class="eckb-article-toc ' . $classes . ' eckb-article-toc-reset "				
				data-offset="' . $args['config']['article_toc_scroll_offset'] . '"
				data-min="' . $args['config']['article_toc_start_level'] . '"
				data-exclude_class="' . $args['config']['article_toc_exclude_class'] . '"
				>' . $title . '</div>
			';

		echo $wrap;
	}

	/**
	 * For Category Focused Layout show top level or sibling categories in the right sidebar
	 *
	 * @param $args
	 */
	public static function focused_layout_categories( $args ) {

		// for Category Focused Layout show sidebar with list of top-level categories
		if ( $args['config']['kb_main_page_layout'] != EPKB_KB_Config_Layout_Categories::LAYOUT_NAME ) {
			return;
		}

		$category_id = 0;
		$active_id = 0;
		$breadcrumb_tree = EPKB_Templates_Various::get_article_breadcrumb( $args['config'], $args['article']->ID );
		$breadcrumb_tree = array_keys( $breadcrumb_tree );
		if ( $args['config']['categories_layout_list_mode'] == 'list_top_categories' ) {
			if ( isset( $breadcrumb_tree[0] ) ) {
				$active_id = $breadcrumb_tree[0];
			}
		} else {
			$tree_count = count( $breadcrumb_tree );
			if ( $tree_count > 1 ) {
				$category_id = $breadcrumb_tree[$tree_count - 2];
				$active_id = $breadcrumb_tree[$tree_count - 1];
			}

			if ( $tree_count == 1 ) {
				$active_id = $breadcrumb_tree[0];
			}
		}

		echo EPKB_Categories_DB::get_layout_categories_list( $args['config']['id'], $args['config'], $category_id, $active_id );
	}

	// CREATED ON
	public static function created_on( $args ) {
		echo '<div class="eckb-ach__article-meta__date-created">';
			if ( 'on' == $args['config']['article_meta_icon_on'] ) {
				echo '<span class="eckb-ach__article-meta__date-created__date-icon epkbfa epkbfa-calendar"></span>';
			}
			if ( $args['config']['created_on_text'] && ! empty($args['article']->post_date) ) {
				echo '<span class="eckb-ach__article-meta__date-created__text">' . esc_html( $args['config']['created_on_text'] ) . '</span>';
			}

			echo '<span class="eckb-ach__article-meta__date-created__date">';
            printf(
                '<time class="entry-date" datetime="%1$s">%2$s</time>',
                esc_attr( get_the_date( DATE_W3C ) ),
                esc_html( get_the_date() )
            );
			echo '</span>';
		echo '</div>';
	}

    // LAST UPDATED ON
    public static function last_updated_on( $args ) {
	    echo '<div class="eckb-ach__article-meta__date-updated">';
			if ( 'on' == $args['config']['article_meta_icon_on'] ) {
				echo '<span class="eckb-ach__article-meta__date-updated__date-icon epkbfa epkbfa-pencil-square-o"></span>';
			}
		    if ( $args['config']['last_udpated_on_text'] && ! empty($args['article']->post_modified) ) {
			    echo '<span class="eckb-ach__article-meta__date-updated__text">' . esc_html( $args['config']['last_udpated_on_text'] ) . '</span>';
		    }

		    echo '<span class="eckb-ach__article-meta__date-updated__date">';
            printf(
                '<time class="entry-date" datetime="%1$s">%2$s</time>',
                esc_attr( get_the_modified_date( DATE_W3C ) ),
                esc_html( get_the_modified_date() )
            );
		    echo '</span>';
	    echo '</div>';
    }

	// AUTHOR
    public static function author( $args ) {

		$post_author = empty($_POST['epkb-wizard-demo-data']) ? get_the_author_meta( 'display_name', $args['article']->post_author ) : 'Admin';

	    echo '<div class="eckb-ach__article-meta__author">';
		    if ( 'on' == $args['config']['article_meta_icon_on'] ) {
				echo '<span class="eckb-ach__article-meta__author__author-icon epkbfa epkbfa-user"></span>';
			}
		    if ( $args['config']['author_text'] && ! empty($post_author) ) {
			    echo '<span class="eckb-ach__article-meta__author__text">' . esc_html( $args['config']['author_text'] ) . '</span>';
		    }
	        echo '<span class="eckb-ach__article-meta__author__name">' . $post_author . '</span>';
	    echo '</div>';
    }

    // BREADCRUMB
    public static function breadcrumbs( $args ) {
        if ( $args['config'][ 'breadcrumb_toggle'] == 'on' ) {
            EPKB_Templates::get_template_part( 'feature', 'breadcrumb', $args['config'], $args['article'] );
        };
    }

    // BACK NAVIGATION
    public static function navigation( $args ) {
        if ( $args['config'][ 'back_navigation_toggle'] == 'on' ) {
            EPKB_Templates::get_template_part( 'feature', 'navigation-back', $args['config'], $args['article'] );
        }
    }

    // ARTICLE CONTENT
    public static function article_content( $args ) {
	    do_action( 'eckb-article-before-content', $args );
            echo $args['content'];
	    do_action( 'eckb-article-after-content', $args );

    }

    // TAGS
    public static function tags( $args ) {
        EPKB_Templates::get_template_part( 'feature', 'tags', $args['config'], $args['article'] );
    }

    // COMMENTS
    public static function comments( $args ) {
        // only show if using our KB template as theme templates display comments
        if ( $args['config'][ 'templates_for_kb' ] == 'kb_templates' && ! self::is_demo_article( $args['article'] ) ) {
            EPKB_Templates::get_template_part( 'feature', 'comments', array(), $args['article'] );
        }
    }

	/**
	 * Disable comments.
	 * Enable comments but it is up to WP, article and theme settings whether comments are actually displayed.
	 *
	 * @param $open
	 * @param $post_id
	 * @return bool
	 */
	public function setup_comments( $open, $post_id ) {

        global $eckb_kb_id;

		// verify it is a KB article
		$post = get_post();
		if ( empty($post) || ! $post instanceof WP_Post || ( ! EPKB_KB_Handler::is_kb_post_type( $post->post_type ) && empty($eckb_kb_id) ) ) {
			return $open;
		}

		$kb_id = empty($eckb_kb_id) ? EPKB_KB_Handler::get_kb_id_from_post_type( $post->post_type ) : $eckb_kb_id;
		if ( is_wp_error($kb_id) ) {
			return $open;
		}

		if ( empty($this->cached_comments_flag) ) {
			$this->cached_comments_flag = epkb_get_instance()->kb_config_obj->get_value( 'articles_comments_global', $kb_id, 'off' );
		}

		return 'on' == $this->cached_comments_flag;
	}

    private static function is_demo_article( $article ) {
        return empty($article->ID) || empty($GLOBALS['post']) || empty($GLOBALS['post']->ID);
    }

	/**
	 * Generate new article VERSION 2 style from configuration
	 *
	 * @param $kb_config
	 */
	private static function generate_article_structure_css_v2($kb_config ) {

		//This controls the whole width of the Article page ( Left Sidebar / Content / Right Sidebar )
		$article_container_width        = $kb_config['article-container-width-v2'];
		$article_container_width_units  = $kb_config['article-container-width-units-v2'];


		// Left Sidebar Settings
		$article_left_sidebar_width     = $kb_config['article-left-sidebar-width-v2'];
		$article_left_sidebar_padding   = $kb_config['article-left-sidebar-padding-v2'];
		$article_left_sidebar_bgColor   = $kb_config['article-left-sidebar-background-color-v2'];

		// Content Settings
		$article_content_width          = $kb_config['article-content-width-v2'];
		$article_content_padding        = $kb_config['article-content-padding-v2'];
		$article_content_bgColor        = $kb_config['article-content-background-color-v2'];

		// Right Sidebar Settings
		$article_right_sidebar_width     = $kb_config['article-right-sidebar-width-v2'];
		$article_right_sidebar_padding   = $kb_config['article-right-sidebar-padding-v2'];
		$article_right_sidebar_bgColor   = $kb_config['article-right-sidebar-background-color-v2'];

		// Advanced
		$mobile_width                    = $kb_config['article-mobile-break-point-v2'];


		$is_left_sidebar_on = // FUTURE if needed: $kb_config['article-left-sidebar-on-v2'] == 'on' ||
		                      ( $kb_config['article_toc_enable'] == 'on' && $kb_config['article_toc_position'] == 'left') ||
		                      $kb_config['kb_sidebar_location'] == 'left-sidebar' ||
		                      $kb_config['kb_article_page_layout'] == EPKB_KB_Config_Layouts::SIDEBAR_LAYOUT ||
		                      $kb_config['kb_main_page_layout'] == EPKB_KB_Config_Layout_Categories::LAYOUT_NAME;

		$is_right_sidebar_on = // FUTURE if needed: $kb_config['article-right-sidebar-on-v2'] == 'on' ||
		                      ( $kb_config['article_toc_enable'] == 'on' && $kb_config['article_toc_position'] == 'right') ||
		                      $kb_config['kb_sidebar_location'] == 'right-sidebar';

		// Deal with Sidebar options.
		/**
		 *  Grid Columns start at lines.
		 *
		 *  Left Sidebar Grid Start:    1 - 2;
		 *  Content Grid Start:         2 - 3;
		 *  Left Sidebar Grid Start:    3 - 4;
		 */

		// If No Left Sidebar Expand the Article Content 1 - 3
		$article_length = '';
		if ( ! $is_left_sidebar_on ) {
			$article_length = '
					#eckb-article-page-container-v2 #eckb-article-body {
					      grid-template-columns:  0 '.$article_content_width.'% '.$article_right_sidebar_width.'%;
					}
					#eckb-article-page-container-v2 #eckb-article-left-sidebar {
							display:none;
					}
					#eckb-article-page-container-v2 #eckb-article-content {
							grid-column-start: 1;
							grid-column-end: 3;
						}
					';
		}

		/**
		 * If No Right Sidebar
		 *  - Expend the Article Content 2 - 4
		 *  - Make Layout 2 Columns only and use the Two remaining values
		 */
		if ( ! $is_right_sidebar_on ) {
			$article_length = '
					#eckb-article-page-container-v2 #eckb-article-body {
					      grid-template-columns: '.$article_left_sidebar_width.'% '.$article_content_width.'% 0 ;
					}
					
					#eckb-article-page-container-v2 #eckb-article-right-sidebar {
							display:none;
					}
					#eckb-article-page-container-v2 #eckb-article-content {
							grid-column-start: 2;
							grid-column-end: 4;
						}
					';
		}

		// If No Sidebars Expand the Article Content 1 - 4
		if ( ! $is_left_sidebar_on && ! $is_right_sidebar_on ) {
			$article_length = '
					#eckb-article-page-container-v2 #eckb-article-body {
					      grid-template-columns: 0'.$article_content_width.'% 0;
					}
					#eckb-article-page-container-v2 #eckb-article-left-sidebar {
							display:none;
					}
					#eckb-article-page-container-v2 #eckb-article-right-sidebar {
							display:none;
					}
					#eckb-article-page-container-v2 #eckb-article-content {
							grid-column-start: 1;
							grid-column-end: 4;
						}
					';
		}

		/**
		 * If Both Sidebars are active
		 *  - Make Layout 3 Columns and divide their sizes according to the user settings
		 */
		if ( $is_left_sidebar_on && $is_right_sidebar_on ) {
			$article_length = '
					#eckb-article-page-container-v2 #eckb-article-body {
					      grid-template-columns: '.$article_left_sidebar_width.'% '.$article_content_width.'% '.$article_right_sidebar_width.'%;
					}
					';
		}		?>

		<!-- Article Structure Version 2 Style -->
		<!--suppress CssUnusedSymbol -->
		<style>
			<?php echo $article_length; ?>
			#eckb-article-page-container-v2 {
				width:<?php echo $article_container_width.$article_container_width_units; ?>;
			}

			#eckb-article-page-container-v2 #eckb-article-left-sidebar {
				padding: <?php echo $article_left_sidebar_padding.'px'; ?>;
				background-color: <?php echo $article_left_sidebar_bgColor; ?>
			}
			#eckb-article-page-container-v2 #eckb-article-content {
				padding: <?php echo $article_content_padding.'px'; ?>;
				background-color: <?php echo $article_content_bgColor; ?>
			}
			#eckb-article-page-container-v2 #eckb-article-right-sidebar {
				padding: <?php echo $article_right_sidebar_padding.'px'; ?>;
				background-color: <?php echo $article_right_sidebar_bgColor; ?>
			}

			/* Media Queries ------------------------------------------------------------*/

			/* Grid Adjust Column sizes for smaller screen */

			/*This option can be in the future */
			/*@media only screen and ( max-width: <?php //echo $tablet_width; ?>px ) {
						#eckb-article-page-container-v2 #eckb-article-body {
							grid-template-columns: 20% 60% 20%;
						}
					}*/

			/*!* Grid becomes 1 Column Layout *!*/
			@media only screen and ( max-width: <?php echo $mobile_width; ?>px ) {

				#eckb-article-page-container-v2 #eckb-article-content {
					grid-column-start: 1;
					grid-column-end: 4;
				}
				#eckb-article-page-container-v2 #eckb-article-left-sidebar {
					grid-column-start: 1;
					grid-column-end: 4;
				}
				#eckb-article-page-container-v2 #eckb-article-right-sidebar {
					grid-column-start: 1;
					grid-column-end: 4;
				}
				#eckb-article-page-container-v2 .eckb-article-toc {
					position: relative;
					float: left;
					width: 100%;
					height: auto;
					top: 0;
				}
			}

		</style>    <?php
	}

	// TODO currently only for Category Layout
	public static function is_article_structure_v2( $kb_config ) {
		$article_version = $kb_config['kb_main_page_layout'] == EPKB_KB_Config_Layout_Categories::LAYOUT_NAME ? 'version-2' : 'version-1';
		return $article_version === 'version-2';
	}
}
