<?php  if ( ! defined( 'ABSPATH' ) ) exit;


/**
 * Display KB configuration Wizard
 *
 * @copyright   Copyright (C) 2018, Echo Plugins
 * @license http://opensource.org/licenses/gpl-2.0.php GNU Public License
 */
class EPKB_KB_Wizard_Cntrl {

	function __construct() {
		add_action( 'wp_ajax_epkb_apply_wizard_changes', array( $this, 'apply_wizard_changes' ) );
		add_action( 'wp_ajax_epkb_wizard_update_color_article_view', array( $this, 'wizard_update_color_article_view' ) );
	}

	public function apply_wizard_changes() {

		// verify that request is authentic
		if ( empty( $_REQUEST['_wpnonce_apply_wizard_changes'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce_apply_wizard_changes'], '_wpnonce_apply_wizard_changes' ) ) {
			EPKB_Utilities::ajax_show_error_die( __( 'You do not have permission to edit this knowledge base', 'echo-knowledge-base' ) );
		}

		// ensure that user has correct permissions
		if ( ! is_admin() || ! current_user_can( 'manage_options' ) ) {
			EPKB_Utilities::ajax_show_error_die( __( 'You do not have permission to edit this knowledge base', 'echo-knowledge-base' ) );
		}

		// get current KB ID
		$wizard_kb_id = EPKB_Utilities::post('epkb_wizard_kb_id');
		if ( empty($wizard_kb_id) || ! EPKB_Utilities::is_positive_int( $wizard_kb_id ) ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Invalid wizard id parameter (2). Please refresh your page', 'echo-knowledge-base' ) );
		}

		// get Wizard type
		$wizard_type = EPKB_Utilities::post('wizard_type');
		if ( empty($wizard_type) ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Invalid wizard type parameter (22). Please refresh your page', 'echo-knowledge-base' ) );
		}

		// get new KB template related configuration
		$new_config_post = EPKB_Utilities::post('kb_config', array());
		if ( empty($new_config_post) || count($new_config_post) < 100 ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Invalid post parameters (1). Please refresh your page', 'echo-knowledge-base' ) );
		}

		// get Wizard type specific filter
		switch( $wizard_type ) {
			case 'theme':
				$wizard_fields = EPKB_KB_Wizard_Themes::$theme_fields;
				break;
			case 'text':
				$wizard_fields = EPKB_KB_Wizard_Text::$text_fields;
				break;
			default:
				EPKB_Utilities::ajax_show_error_die( __( 'Invalid parameters - Wizard type (112). Please refresh your page', 'echo-knowledge-base' ) );
				return;
		}

		// filter fields from Wizard to ensure we are saving only configuration that is applicable for this Wizard
		$new_config = array();
		foreach($new_config_post as $field_name => $field_value) {
			if ( in_array($field_name, $wizard_fields) ) {
				$new_config[$field_name] = $field_value;
			}
		}

		// get current KB configuration (for blank KB, configuration will contain default values)
		$orig_config = epkb_get_instance()->kb_config_obj->get_kb_config( $wizard_kb_id );
		if ( is_wp_error( $orig_config ) ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Error occurred. Please contact us. (8)', 'echo-knowledge-base' ) );
		}

		// get current KB configuration (for blank KB, configuration will contain default values)
		$orig_config = apply_filters( 'epkb_all_wizards_get_current_config', $orig_config, $wizard_kb_id );
		if ( empty($orig_config) || count($orig_config) < 3 ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Invalid parameters (111). Please refresh your page', 'echo-knowledge-base' ) );
		}

		// overwrite current KB configuration with new configuration from this Wizard
		$new_config = array_merge($orig_config, $new_config);

		$is_blank_kb = EPKB_Utilities::is_blank_KB( $wizard_kb_id );
		if ( is_wp_error($is_blank_kb) ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Could not load KB config (3). Please contact us.', 'echo-knowledge-base' ) );
		}

		// call Wizard type specific saving function
		switch( $wizard_type ) {
			case 'theme':
				$this->apply_theme_wizard_changes( $wizard_kb_id, $orig_config, $new_config, $is_blank_kb );
				break;
			case 'text':
				$this->apply_text_wizard_changes( $orig_config, $new_config, $is_blank_kb );
				break;
			default:
				EPKB_Utilities::ajax_show_error_die( __( 'Invalid parameters - Wizard type (112). Please refresh your page', 'echo-knowledge-base' ) );
				return;
		}
	}

	/**
	 * Apply THEME WIZARD changes
	 *
	 * @param $wizard_kb_id
	 * @param $orig_config
	 * @param $new_config
	 * @param $is_blank_kb
	 */
	private function apply_theme_wizard_changes( $wizard_kb_id, $orig_config, $new_config, $is_blank_kb ) {

		// verify correct Article Page layout based on Main Page layout
		$article_page_layouts = EPKB_KB_Config_Layouts::get_article_page_layouts( $new_config['kb_main_page_layout'] );
		if ( empty($article_page_layouts) ) {
			$new_config['kb_article_page_layout'] = EPKB_KB_Config_Layouts::KB_ARTICLE_PAGE_NO_LAYOUT;
		} else if ( ! in_array( $new_config['kb_article_page_layout'], array_keys($article_page_layouts) ) ) {
			$article_pg_layouts = array_keys($article_page_layouts);
			$new_config['kb_article_page_layout'] = $article_pg_layouts[0];
		}

		// get and sanitize KB name
		$kb_name = EPKB_Utilities::post('kb_name');
		$kb_name = empty($kb_name) ? '' : substr( $kb_name, 0, 50 );
		$kb_name = sanitize_text_field($kb_name);
		if ( empty($kb_name) ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Invalid parameters (3). Please refresh your page', 'echo-knowledge-base' ) );
		}

		// do we need to create a new KB content?
		if ( $is_blank_kb ) {

			// 1. save KB configuration based on Wizard changes
			$new_config['status'] = EPKB_KB_Status::BLANK;
			$orig_config = EPKB_KB_Config_Specs::get_default_kb_config( $wizard_kb_id );
			$orig_config['status'] = EPKB_KB_Status::BLANK;
			$udpate_kb_msg = $this->update_kb_configuration( $wizard_kb_id, $orig_config, $new_config );
			if ( ! empty($udpate_kb_msg) ) {
				EPKB_Utilities::ajax_show_error_die( __( 'Error occurred. Could not create KB (34). ', 'echo-knowledge-base' ) . EPKB_Utilities::contact_us_for_support() );
			}

			// get KB slug
			$kb_slug = EPKB_Utilities::post('kb_slug');
			$kb_slug = empty($kb_slug) ? '' : substr( $kb_slug, 0, 100 );
			$kb_slug = sanitize_title($kb_slug);
			if ( $is_blank_kb && empty($kb_slug) ) {
				EPKB_Utilities::ajax_show_error_die( __( 'Invalid parameters (4). Please refresh your page', 'echo-knowledge-base' ) );
			}

			// 2. add sample content and Main Page
			$new_config = EPKB_KB_Handler::add_new_knowledge_base( $wizard_kb_id, $kb_name, $kb_slug );
			if ( is_wp_error($new_config) ) {
				EPKB_Utilities::ajax_show_error_die( __( 'Error occurred. Could not create KB (35). ', 'echo-knowledge-base' ) . EPKB_Utilities::contact_us_for_support() );
			}
			
		} else {

			// prevent new config to overwrite essential fields
			$new_config['id'] = $orig_config['id'];
			$new_config['status'] = $orig_config['status'];
			$new_config['kb_name'] = $kb_name;
			$new_config['kb_main_pages'] = $orig_config['kb_main_pages'];
			$new_config['kb_articles_common_path'] = $orig_config['kb_articles_common_path'];

			// update KB and add-ons configuration
			$update_kb_msg = $this->update_kb_configuration( $wizard_kb_id, $orig_config, $new_config );
			if ( ! empty($update_kb_msg) ) {
				EPKB_Utilities::ajax_show_error_die( __( 'Error occurred. Could not create KB (36). ', 'echo-knowledge-base' ) . EPKB_Utilities::contact_us_for_support() );
			}
		}

		// add items to menus if needs
		$menu_ids = EPKB_Utilities::post( 'menu_ids', array(), false );
		if ( $menu_ids && ! empty($new_config['kb_main_pages']) ) {
			$kb_main_pages = $new_config['kb_main_pages'];
			foreach ( $menu_ids as $id ) {
				$itemData =  array(
					'menu-item-object-id'   => key($kb_main_pages),
					'menu-item-parent-id'   => 0,
					'menu-item-position'    => 99,
					'menu-item-object'      => 'page',
					'menu-item-type'        => 'post_type',
					'menu-item-status'      => 'publish'
				  );

				wp_update_nav_menu_item( $id, 0, $itemData );
			}
		}

		// in case user changed article common path, flush the rules
		EPKB_Articles_CPT_Setup::register_custom_post_type( $new_config, $new_config['id'] );

		// always flush the rules; this will ensure that proper rewrite rules for layouts with article visible will be added
		flush_rewrite_rules( false );
		update_option('epkb_flush_rewrite_rules', true);

		$message = __('Configuration Saved', 'echo-knowledge-base');
		wp_die( json_encode( array( 'message' => $message, 'kb_main_page_link' => EPKB_KB_Handler::get_first_kb_main_page_url( $new_config ) ) ) );
	}

	/**
	 * Apply TEXT WIZARD changes
	 *
	 * @param $orig_config
	 * @param $new_config
	 * @param $is_blank_kb
	 */
	private function apply_text_wizard_changes( $orig_config, $new_config, $is_blank_kb ) {

		// KB should not be blank
		if ( $is_blank_kb ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Could not process KB config (3). Please contact us.', 'echo-knowledge-base' ) );
		}

		// prevent new config to overwrite essential fields
		$new_config['id'] = $orig_config['id'];
		$new_config['status'] = $orig_config['status'];
		$new_config['kb_main_pages'] = $orig_config['kb_main_pages'];
		$new_config['kb_articles_common_path'] = $orig_config['kb_articles_common_path'];

		// update KB and add-ons configuration
		$update_kb_msg = $this->update_kb_configuration( $orig_config['id'], $orig_config, $new_config );
		if ( ! empty($update_kb_msg) ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Error occurred. Could not create KB (36). ', 'echo-knowledge-base' ) . EPKB_Utilities::contact_us_for_support() );
		}

		$message = __('Text Configuration Saved', 'echo-knowledge-base');
		wp_die( json_encode( array( 'message' => $message, 'kb_main_page_link' => EPKB_KB_Handler::get_first_kb_main_page_url( $new_config ) ) ) );
	}

	/**
	 * Triggered when user submits changes to KB configuration
	 *
	 * @param $kb_id
	 * @param $orig_config
	 * @param $new_config
	 * @return string
	 */
	 // TODO if future: refractor this function and the same in kb-config-controller
	public function update_kb_configuration( $kb_id, $orig_config, $new_config ) {

		// core handles only default KB
		if ( $kb_id != EPKB_KB_Config_DB::DEFAULT_KB_ID && ! defined( 'E' . 'MKB_PLUGIN_NAME' ) ) {
			return __('Ensure that Multiple KB add-on is active and refresh this page', 'echo-knowledge-base');
		}
		
		// sanitize all fields in POST message
		$field_specs = EPKB_KB_Config_Controller::retrieve_all_kb_specs( $kb_id );
		$form_fields = EPKB_Utilities::retrieve_and_sanitize_form( $new_config, $field_specs );
		if ( empty($form_fields) ) {
			EPKB_Logging::add_log("form fields missing");
			return __( 'Form fields missing. Please refresh your browser', 'echo-knowledge-base' );
		} else if ( count($form_fields) < 100 ) {
			return __( 'Some form fields are missing. Please refresh your browser and try again or contact support', 'echo-knowledge-base' );
		}

		// sanitize fields based on each field type
		$input_handler = new EPKB_Input_Filter();
		$new_kb_config = $input_handler->retrieve_and_sanitize_form_fields( $form_fields, $field_specs, $orig_config );

		// save add-ons configuration
		$form_fields['icons_not_saved'] = true;
		$result = apply_filters( 'epkb_kb_config_save_input_v2', '', $kb_id, $form_fields, $new_kb_config['kb_main_page_layout'] );
		if ( is_wp_error( $result ) ) {
			/* @var $result WP_Error */
			$message = $result->get_error_data();
			if ( empty($message) ) {
				return $result->get_error_message();
			} else {
				return $message;
			}
		}

		// if Sidebar layout chosen and TOC is on the left, set it to the right
		// TODO remove?
		if ( $new_kb_config['kb_article_page_layout'] == EPKB_KB_Config_Layouts::SIDEBAR_LAYOUT && $new_kb_config['article_toc_position'] == 'left' ) {
			$new_kb_config['article_toc_position'] = 'right';
		}

		// ensure kb id is preserved
		$new_kb_config['id'] = $kb_id;

		// TODO for now save previous configuration
		EPKB_Utilities::save_kb_option( $kb_id, 'epkb_orignal_config', $orig_config, true );

		// save KB core configuration
		$result = epkb_get_instance()->kb_config_obj->update_kb_configuration( $kb_id, $new_kb_config );
		if ( is_wp_error( $result ) ) {
			/* @var $result WP_Error */
			$message = $result->get_error_data();
			if ( empty($message) ) {
				return __( 'Could not save the new configuration (3)', 'echo-knowledge-base' );
			} else {
				return __( 'Configuration NOT saved (4) due to following problems:', 'echo-knowledge-base' );
			}
		}

		// we are done here
		return '';
	}

	/**
	 * Article theme layout has changed so update the article preview.
	 */
	public function wizard_update_color_article_view() {
		if ( empty( $_REQUEST['_wpnonce_apply_wizard_changes'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce_apply_wizard_changes'], '_wpnonce_apply_wizard_changes' ) ) {
			EPKB_Utilities::ajax_show_error_die( __( 'You do not have permission to edit this knowledge base', 'echo-knowledge-base' ) );
		}
		
		// get new KB config
		$new_config_post = EPKB_Utilities::post('kb_config', array());
		if ( empty($new_config_post) || count($new_config_post) < 100 ) {
			EPKB_Utilities::ajax_show_error_die( __( 'Invalid parameters (10). Please refresh your page', 'echo-knowledge-base' ) );
		}

		$_POST['epkb-wizard-demo-data'] = true;
		$_POST['epkb_demo_kb'] = true;
		
		$orig_config = epkb_get_instance()->kb_config_obj->get_kb_config( EPKB_KB_Config_DB::DEFAULT_KB_ID );
		$new_config = array_merge($orig_config, $new_config_post);
		$handler = new EPKB_KB_Config_Page( $new_config );
		wp_die( json_encode( array( 'message' => $new_config, 'html' => $handler->display_article_page_layout_preview( false ) ) ) );
	}
}
